import datetime
import re

from winacl.dtyp.sid import SID
from winacl.dtyp.guid import GUID
from winacl.dtyp.security_descriptor import SECURITY_DESCRIPTOR
from msldap import logger
from msldap.protocol.messages import Attribute, Change, PartialAttribute
from msldap.wintypes.managedpassword import MSDS_MANAGEDPASSWORD_BLOB

MSLDAP_DT_WIN_EPOCH = datetime.datetime(1601, 1, 1)

#this regex and the function that uses it is from ldap3
time_format = re.compile(
        r'''
        ^
        (?P<Year>[0-9]{4})
        (?P<Month>0[1-9]|1[0-2])
        (?P<Day>0[1-9]|[12][0-9]|3[01])
        (?P<Hour>[01][0-9]|2[0-3])
        (?:
          (?P<Minute>[0-5][0-9])
          (?P<Second>[0-5][0-9]|60)?
        )?
        (?:
          [.,]
          (?P<Fraction>[0-9]+)
        )?  
        (?:
          Z
          |
          (?:
            (?P<Offset>[+-])
            (?P<OffHour>[01][0-9]|2[0-3])
            (?P<OffMinute>[0-5][0-9])?
          )
        )
        $
        ''',
        re.VERBOSE
    )

def bytes2timedelta(x, encode = False):
	return int2timedelta([int.from_bytes(x[0], byteorder='little', signed=True)], encode=encode)

def int2timedelta(x, encode = False):
	if encode is False:
		x = int(x[0])
		if x == '-9223372036854775808':
			return datetime.timedelta.max
		return datetime.timedelta(microseconds=(x / 10.))
	raise NotImplementedError()

###### NEW FUNCTIONS

def single_utf16le(x, encode = False):
	if encode is False:
		return x[0].decode('utf-16-le')
	return [x.encode('utf-16-le')]

def single_interval(x, encode = False):
	if encode is False:
		x = int(x[0])
		if x == 9223372036854775807:
			return datetime.datetime.max.replace(tzinfo=datetime.timezone.utc)
		
		us = x / 10.
		try:
			return (MSLDAP_DT_WIN_EPOCH + datetime.timedelta(microseconds=us)).replace(tzinfo=datetime.timezone.utc)
		except:
			return MSLDAP_DT_WIN_EPOCH
	raise NotImplementedError()

def x2gmsa(x, encode = False):
	if encode is False:
		return MSDS_MANAGEDPASSWORD_BLOB.from_bytes(x[0])
	raise NotImplementedError()

def single_sid(x, encode = False):
	if encode is False:
		return str(SID.from_bytes(x[0]))
	return [SID.from_string(x).to_bytes()]

def single_str(x, encode = False, encoding = 'utf-8'):
	if encode is False:
		return x[0].decode(encoding)
	return [x.encode(encoding)]

def single_bytes(x, encode = False):
	if encode is False:
		return x[0]
	return [x]

def single_int(x, encode = False):
	if encode is False:
		return int(x[0])
	return [str(x).encode()]

def single_bool(x, encode = False, encoding = 'utf-8'):
	if encode is False:
		x = x[0].decode(encoding)
		if x == 'FALSE':
			return False
		return True
	return [b'TRUE' if x is True else b'FALSE']

def single_guid(x, encode = False):
	if encode is False:
		return str(GUID.from_bytes(x[0]))
	return [GUID.from_string(x).to_bytes()]

def single_sd(x, encode = False):
	if encode is False:
		return SECURITY_DESCRIPTOR.from_bytes(x[0])
	return [x.to_bytes()]

def single_date(x, encode = False, encoding = 'utf-8'):
	if encode is False:
		try:
			x = x[0].decode(encoding)
			match = time_format.fullmatch(x)
			if match is None:
				return x
			matches = match.groupdict()

			offset = datetime.timedelta(
				hours=int(matches['OffHour'] or 0),
				minutes=int(matches['OffMinute'] or 0)
			)

			if matches['Offset'] == '-':
				offset *= -1

			# Python does not support leap second in datetime (!)
			if matches['Second'] == '60':
				matches['Second'] = '59'

			# According to RFC, fraction may be applied to an Hour/Minute (!)
			fraction = float('0.' + (matches['Fraction'] or '0'))

			if matches['Minute'] is None:
				fraction *= 60
				minute = int(fraction)
				fraction -= minute
			else:
				minute = int(matches['Minute'])

			if matches['Second'] is None:
				fraction *= 60
				second = int(fraction)
				fraction -= second
			else:
				second = int(matches['Second'])

			microseconds = int(fraction * 1000000)

			return datetime.datetime(
				int(matches['Year']),
				int(matches['Month']),
				int(matches['Day']),
				int(matches['Hour']),
				minute,
				second,
				microseconds,
				datetime.timezone(offset),
			)
		except Exception:  # exceptions should be investigated, anyway the formatter return the raw_value
			pass
		return x
	raise NotImplementedError()

def multi_date(x, encode = False):
	if encode is False:
		res = []
		for a in x:
			res.append(single_date((a, None)))
		return res
	if isinstance(x, list) is False:
		x = [x]
	raise NotImplementedError()

def multi_int(x, encode = False):
	if encode is False:
		return [int(e) for e in x ]
	if isinstance(x, list) is False:
		x = [x]
	return [str(k).encode() for k in x]

def multi_str(x, encode = False, encoding = 'utf-8'):
	if encode is False:
		return [e.decode(encoding) for e in x ]
	if isinstance(x, list) is False:
		x = [x]
	return [line.encode(encoding) for line in x]

def multi_sid(x, encode = False):
	if encode is False:
		t = []
		for s in x:
			t.append(str(SID.from_bytes(s)))
		return t
	if isinstance(x, list) is False:
		x = [x]
	return [SID.from_string(k).to_bytes() for k in x]

def multi_interval(x, encode = False):
	if encode is False:
		t = []
		for s in x:
			t.append(single_interval(s))
		return t
	if isinstance(x, list) is False:
		x = [x]
	raise NotImplementedError()

def multi_sid(x, encode = False):
	if encode is False:
		t = []
		for s in x:
			t.append(str(SID.from_bytes(s)))
		return t
	if isinstance(x, list) is False:
		x = [x]
	return [SID.from_string(k).to_bytes() for k in x]

def multi_bytes(x, encode=False):
	if encode is False:
		return x
	if isinstance(x, list) is False:
		x = [x]
	return x

def multi_bool(x, encode=False):
	if encode is False:
		return [single_bool([r]) for r in x]
	if isinstance(x, list) is False:
		x = [x]
	return [single_bool(r)[0] for r in x]

def multi_sd(x, encode=False):
	if encode is False:
		return [single_sd([r] for r in x)]
	if isinstance(x, list) is False:
		x = [x]
	return [single_sd(r)[0] for r in x]

MSLDAP_BUILTIN_ATTRIBUTE_TYPES = {
	'supportedCapabilities' : multi_str,
	'serverName' : single_str,
	'ldapServiceName': single_str,
	'dnsHostName' : single_str,
	'supportedSASLMechanisms' : multi_str,
	'supportedLDAPPolicies' : multi_str,
	'supportedLDAPVersion' : multi_int,
	'supportedControl' : multi_str,
	'rootDomainNamingContext' : single_str,
	'configurationNamingContext' : single_str,
	'schemaIDGUID' : single_guid,
	'lDAPDisplayName' : single_str, 
	'schemaNamingContext' : single_str,
	'defaultNamingContext' : single_str,
	'adminDescription' : single_str,
	'adminDisplayName' : single_str,
	'namingContexts' : multi_str,
	'dsServiceName' : single_str,
	'subschemaSubentry' : single_str,
	'distinguishedName' : single_str,
	'objectCategory' : single_str,
	'userPrincipalName' : single_str,
	'sAMAccountType' : single_int,
	'sAMAccountName' : single_str,
	'logonCount' : single_int,
	'accountExpires' : single_interval,
	'objectSid' : single_sid,
	'primaryGroupID' : single_int,
	'pwdLastSet' : single_interval,
	'lastLogon' : single_interval,
	'lastLogoff' : single_interval,
	'lastLogonTimestamp' : single_interval,
	'badPasswordTime' : single_interval,
	'countryCode' : single_int,
	'codePage' : single_int,
	'badPwdCount' : single_int,
	'userAccountControl' : single_int,
	'objectGUID' : single_guid,
	'name' : single_str,
	'displayName' : single_str,
	'whenChanged' : single_date,
	'whenCreated' : single_date,
	'distinguishedName' : single_str,
	'givenName' : single_str,
	'cn' : single_str,
	'objectClass' : multi_str,
	'dc' : single_str,
	'msDS-Behavior-Version' : single_int,
	'masteredBy' : single_str,
	'systemFlags' : single_int,
	'rIDManagerReference' : single_str,
	'auditingPolicy' : single_bytes,
	'uASCompat' : single_int,
	'serverState' : single_int,
	'nextRid' : single_int,
	'minPwdLength' : single_int,
	'minPwdAge' : int2timedelta,
	'lockoutThreshold' : single_int,
	'lockOutObservationWindow' : single_int,
	'lockoutDuration' : single_int,
	'forceLogoff' : int2timedelta,
	'creationTime' : single_interval,
	'maxPwdAge' : int2timedelta,
	'pwdHistoryLength' : single_int,
	'pwdProperties' : single_int,
	'uSNChanged' : single_int,
	'uSNCreated' : single_int,
	'instanceType' : single_int,
	'memberOf' : multi_str,
	'description' : single_str,
	'servicePrincipalName' : multi_str,
	'sn' : single_str,
	'initials' : single_str,
	#'nTSecurityDescriptor' : x2sd,
	'nTSecurityDescriptor' : single_bytes,
	'tokenGroups' : multi_sid,
	'localPolicyFlags' : single_int,
	'msDS-SupportedEncryptionTypes' : single_int,
	'isCriticalSystemObject' : single_bool,
	'dNSHostName' : single_str,
	'operatingSystemVersion' : single_str,
	'operatingSystem' : single_str,
	'ou' : single_str,
	'showInAdvancedViewOnly' : single_bool,
	'gPLink' : single_str,
	'gPCFileSysPath' : single_str,
	'flags' : single_int,
	'versionNumber' : single_int,
	'gPCFunctionalityVersion' : single_int,
	'gPCMachineExtensionNames' : multi_str,
	'gPCUserExtensionNames' : multi_str,
	'groupType' : single_int,
	'member' : multi_str,
	'adminCount' : single_int,
	'msDS-AllowedToDelegateTo' : multi_str,
	'dSCorePropagationData' : single_date,
	'trustDirection' : single_int,
	'trustType' : single_int,
	'trustAttributes' : single_int,
	'flatName' : single_str,
	'trustPosixOffset' : single_int,
	'trustPartner' : single_str,
	'securityIdentifier' : single_bytes,
	'unicodePwd' : single_str,
	'ms-Mcs-AdmPwd' : single_str,
	'msLAPS-EncryptedPassword': single_bytes,
	'msLAPS-EncryptedPasswordHistory': multi_bytes,
	'msLAPS-EncryptedDSRMPassword': single_bytes,
	'msLAPS-EncryptedDSRMPasswordHistory': multi_bytes,
	'msLAPS-Password': single_str,
	'msLAPS-PasswordExpirationTime': single_interval,
	'msDS-AllowedToActOnBehalfOfOtherIdentity': single_bytes,
	'msLAPS-Encrypted-Password-Attributes': single_bytes,
	'cACertificate': single_bytes,
	'certificateTemplates': multi_str,
	'cACertificateDN': single_str,
	'msPKI-Enrollment-Servers': multi_str, # it's actually a 5-part multi string split by '\n'
	'revision' : single_int,
	'pKIKeyUsage' : single_bytes,
	'pKIDefaultKeySpec' : single_str,
	'pKIMaxIssuingDepth' : single_int,
	'pKICriticalExtensions' : single_str,
	'pKIExpirationPeriod' : bytes2timedelta,
	'pKIOverlapPeriod' : bytes2timedelta,
	'pKIExtendedKeyUsage' : multi_str,
	'msPKI-RA-Signature' : single_int,
	'msPKI-Enrollment-Flag' : single_int,
	'msPKI-Private-Key-Flag' : single_int,
	'msPKI-Certificate-Name-Flag' : single_int,
	'msPKI-Minimal-Key-Size' : single_int,
	'msPKI-Template-Schema-Version': single_int, 
    'msPKI-Template-Minor-Revision': single_int, 
	'msPKI-Cert-Template-OID' : single_str,
	'msPKI-Certificate-Application-Policy' : multi_str,
	'msPKI-RA-Application-Policies' : multi_str, #I'm guessing here
	'msDS-ManagedPassword' : x2gmsa,
	'msDS-GroupMSAMembership' : single_sd,
	'isSingleValued' : single_bool,
	'attributeSyntax' : single_str, #na
	'isMemberOfPartialAttributeSet' : single_bool, #na
	'attributeID': single_str, #na
	'gidNumber' : single_int,
	'msds-additionaldnshostname' : multi_str,
}

MSLDAP_BUILTIN_ATTRIBUTE_TYPES_ENC = {
	#'objectClass' : multi_str,
	#'sn' : single_str,
	#'gidNumber' : single_int,
	'unicodePwd' : single_utf16le,
	#'lockoutTime' : single_int,
	#'sAMAccountName' : single_str,
	#'userAccountControl' : single_int,
	#'displayName' : single_str,
	#'userPrincipalName' : single_str,
	#'servicePrincipalName' : multi_str,
	#'msds-additionaldnshostname' : multi_str,
	#'gPCMachineExtensionNames' : multi_str,
	#'gPCUserExtensionNames' : multi_str,
	#'versionNumber' : single_int,
	#'member' : multi_str,
	'msDS-AllowedToActOnBehalfOfOtherIdentity' : single_bytes,
	'nTSecurityDescriptor' : single_bytes,
	#'msPKI-Certificate-Name-Flag' : single_int
}

def encode_attributes(x):
	"""converts a dict to attributelist"""
	res = []
	for k in x:
		lookup_table = None
		if k in MSLDAP_BUILTIN_ATTRIBUTE_TYPES_ENC:
			lookup_table = MSLDAP_BUILTIN_ATTRIBUTE_TYPES_ENC
		elif k in MSLDAP_BUILTIN_ATTRIBUTE_TYPES:
			lookup_table = MSLDAP_BUILTIN_ATTRIBUTE_TYPES
		else:
			raise Exception('Unknown conversion type for key "%s"' % k)
		
		res.append(Attribute({
			'type' : k.encode(),
			'attributes' : lookup_table[k](x[k], True)
		}))

	return res

def convert_attributes(x):
	t = {}
	for e in x:
		#print(e)
		k = e['type'].decode()
		#print('k: %s' % k)
		
		if k in MSLDAP_BUILTIN_ATTRIBUTE_TYPES:
			t[k] = MSLDAP_BUILTIN_ATTRIBUTE_TYPES[k](e['attributes'], False)
		elif k in LDAP_WELL_KNOWN_ATTRS:
			t[k] = LDAP_WELL_KNOWN_ATTRS[k](e['attributes'], False)
		else:
			logger.debug('Unknown type! %s data: %s' % (k, e['attributes']))
			t[k] = e['attributes']
	return t

def convert_result(x):
	#print(x)
	#import traceback
	#traceback.print_stack()
	return {
		'objectName' : x['objectName'].decode(),
		'attributes' : convert_attributes(x['attributes'])
	}


def encode_changes(x):
	logger.debug('Encode changes: %s' % x)
	res = []
	for k in x:
		lookup_table = None
		if k in MSLDAP_BUILTIN_ATTRIBUTE_TYPES_ENC:
			lookup_table = MSLDAP_BUILTIN_ATTRIBUTE_TYPES_ENC
		elif k in MSLDAP_BUILTIN_ATTRIBUTE_TYPES:
			lookup_table = MSLDAP_BUILTIN_ATTRIBUTE_TYPES
		else:
			raise Exception('Unknown conversion type for key "%s"' % k)
		
		for mod, value in x[k]:
			res.append(Change({
				'operation' : mod,
				'modification' : PartialAttribute({
					'type' : k.encode(),
					'attributes' : lookup_table[k](value, True)
				})
			}))
			#print(lookup_table[k](value, True))
	return res

LDAP_WELL_KNOWN_ATTRS = {
	"accountExpires" : single_int,
	"accountNameHistory" : multi_str,
	"aCSAggregateTokenRatePerUser" : single_int,
	"aCSAllocableRSVPBandwidth" : single_int,
	"aCSCacheTimeout" : single_int,
	"aCSDirection" : single_int,
	"aCSDSBMDeadTime" : single_int,
	"aCSDSBMPriority" : single_int,
	"aCSDSBMRefresh" : single_int,
	"aCSEnableACSService" : single_bool,
	"aCSEnableRSVPAccounting" : single_bool,
	"aCSEnableRSVPMessageLogging" : single_bool,
	"aCSEventLogLevel" : single_int,
	"aCSIdentityName" : multi_str,
	"aCSMaxAggregatePeakRatePerUser" : single_int,
	"aCSMaxDurationPerFlow" : single_int,
	"aCSMaxNoOfAccountFiles" : single_int,
	"aCSMaxNoOfLogFiles" : single_int,
	"aCSMaxPeakBandwidth" : single_int,
	"aCSMaxPeakBandwidthPerFlow" : single_int,
	"aCSMaxSizeOfRSVPAccountFile" : single_int,
	"aCSMaxSizeOfRSVPLogFile" : single_int,
	"aCSMaxTokenBucketPerFlow" : single_int,
	"aCSMaxTokenRatePerFlow" : single_int,
	"aCSMaximumSDUSize" : single_int,
	"aCSMinimumDelayVariation" : single_int,
	"aCSMinimumLatency" : single_int,
	"aCSMinimumPolicedSize" : single_int,
	"aCSNonReservedMaxSDUSize" : single_int,
	"aCSNonReservedMinPolicedSize" : single_int,
	"aCSNonReservedPeakRate" : single_int,
	"aCSNonReservedTokenSize" : single_int,
	"aCSNonReservedTxLimit" : single_int,
	"aCSNonReservedTxSize" : single_int,
	"aCSPermissionBits" : single_int,
	"aCSPolicyName" : single_str,
	"aCSPriority" : single_int,
	"aCSRSVPAccountFilesLocation" : single_str,
	"aCSRSVPLogFilesLocation" : single_str,
	"aCSServerList" : multi_str,
	"aCSServiceType" : single_int,
	"aCSTimeOfDay" : single_str,
	"aCSTotalNoOfFlows" : single_int,
	"notes" : single_str,
	"additionalTrustedServiceNames" : multi_str,
	"streetAddress" : single_str,
	"addressBookRoots" : multi_str,
	"addressBookRoots2" : multi_str,
	"addressEntryDisplayTable" : single_bytes,
	"addressEntryDisplayTableMSDOS" : single_bytes,
	"homePostalAddress" : single_str,
	"addressSyntax" : single_bytes,
	"addressType" : single_str,
	"adminContextMenu" : multi_str,
	"adminCount" : single_int,
	"adminDescription" : single_str,
	"adminDisplayName" : single_str,
	"adminMultiselectPropertyPages" : multi_str,
	"adminPropertyPages" : multi_str,
	"allowedAttributes" : multi_str,
	"allowedAttributesEffective" : multi_str,
	"allowedChildClasses" : multi_str,
	"allowedChildClassesEffective" : multi_str,
	"altSecurityIdentities" : multi_str,
	"aNR" : single_str,
	"appSchemaVersion" : single_int,
	"applicationName" : single_str,
	"appliesTo" : multi_str,
	"assetNumber" : single_str,
	"assistant" : single_str,
	"assocNTAccount" : single_bytes,
	"associatedDomain" : multi_str,
	"associatedName" : multi_str,
	"attributeDisplayNames" : multi_str,
	"attributeID" : single_str,
	"attributeSecurityGUID" : single_bytes,
	"attributeSyntax" : single_str,
	"attributeTypes" : multi_str,
	"attributeCertificateAttribute" : multi_bytes,
	"audio" : multi_bytes,
	"auditingPolicy" : single_bytes,
	"authenticationOptions" : single_int,
	"authorityRevocationList" : multi_bytes,
	"auxiliaryClass" : multi_str,
	"badPasswordTime" : single_int,
	"badPwdCount" : single_int,
	"birthLocation" : single_bytes,
	"bootFile" : multi_str,
	"bootParameter" : multi_str,
	"bridgeheadServerListBL" : multi_str,
	"bridgeheadTransportList" : multi_str,
	"buildingName" : multi_str,
	"builtinCreationTime" : single_int,
	"builtinModifiedCount" : single_int,
	"businessCategory" : multi_str,
	"bytesPerMinute" : single_int,
	"cACertificate" : multi_bytes,
	"cACertificateDN" : single_str,
	"cAConnect" : single_str,
	"cAUsages" : multi_str,
	"cAWEBURL" : single_str,
	"canUpgradeScript" : multi_str,
	"canonicalName" : multi_str,
	"carLicense" : multi_str,
	"catalogs" : multi_str,
	"categories" : multi_str,
	"categoryId" : single_bytes,
	"certificateAuthorityObject" : single_str,
	"certificateRevocationList" : single_bytes,
	"certificateTemplates" : multi_str,
	"classDisplayName" : multi_str,
	"codePage" : single_int,
	"cOMClassID" : multi_str,
	"cOMCLSID" : single_str,
	"cOMInterfaceID" : multi_str,
	"cOMOtherProgId" : multi_str,
	"cOMProgID" : multi_str,
	"cOMTreatAsClassId" : single_str,
	"cOMTypelibId" : multi_str,
	"cOMUniqueLIBID" : single_str,
	"info" : single_str,
	"cn" : single_str,
	"company" : single_str,
	"contentIndexingAllowed" : single_bool,
	"contextMenu" : multi_str,
	"controlAccessRights" : multi_bytes,
	"cost" : single_int,
	"countryCode" : single_int,
	"c" : single_str,
	"createDialog" : single_str,
	"createTimeStamp" : single_date,
	"createWizardExt" : multi_str,
	"creationTime" : single_int,
	"creationWizard" : single_str,
	"creator" : single_str,
	"cRLObject" : single_str,
	"cRLPartitionedRevocationList" : single_bytes,
	"crossCertificatePair" : multi_bytes,
	"currMachineId" : single_bytes,
	"currentLocation" : single_bytes,
	"currentParentCA" : multi_str,
	"currentValue" : single_bytes,
	"dBCSPwd" : single_bytes,
	"defaultClassStore" : multi_str,
	"defaultGroup" : single_str,
	"defaultHidingValue" : single_bool,
	"defaultLocalPolicyObject" : single_str,
	"defaultObjectCategory" : single_str,
	"defaultPriority" : single_int,
	"defaultSecurityDescriptor" : single_str,
	"deltaRevocationList" : multi_bytes,
	"department" : single_str,
	"departmentNumber" : multi_str,
	"description" : multi_str,
	"desktopProfile" : single_str,
	"destinationIndicator" : multi_str,
	"dhcpClasses" : multi_bytes,
	"dhcpFlags" : single_int,
	"dhcpIdentification" : single_str,
	"dhcpMask" : multi_str,
	"dhcpMaxKey" : single_int,
	"dhcpObjDescription" : single_str,
	"dhcpObjName" : single_str,
	"dhcpOptions" : multi_bytes,
	"dhcpProperties" : multi_bytes,
	"dhcpRanges" : multi_str,
	"dhcpReservations" : multi_str,
	"dhcpServers" : multi_str,
	"dhcpSites" : multi_str,
	"dhcpState" : multi_str,
	"dhcpSubnets" : multi_str,
	"dhcpType" : single_int,
	"dhcpUniqueKey" : single_int,
	"dhcpUpdateTime" : single_int,
	"displayName" : single_str,
	"displayNamePrintable" : single_str,
	"dITContentRules" : multi_str,
	"division" : single_str,
	"dMDLocation" : single_str,
	"dmdName" : single_str,
	"dNReferenceUpdate" : multi_str,
	"dnsAllowDynamic" : single_bool,
	"dnsAllowXFR" : single_bool,
	"dNSHostName" : single_str,
	"dnsNotifySecondaries" : multi_int,
	"dNSProperty" : multi_bytes,
	"dnsRecord" : multi_bytes,
	"dnsRoot" : multi_str,
	"dnsSecureSecondaries" : multi_int,
	"dNSTombstoned" : single_bool,
	"documentAuthor" : multi_str,
	"documentIdentifier" : multi_str,
	"documentLocation" : multi_str,
	"documentPublisher" : multi_str,
	"documentTitle" : multi_str,
	"documentVersion" : multi_str,
	"domainCAs" : multi_str,
	"dc" : single_str,
	"domainCrossRef" : single_str,
	"domainID" : single_str,
	"domainIdentifier" : single_int,
	"domainPolicyObject" : single_str,
	"domainPolicyReference" : single_str,
	"domainReplica" : single_str,
	"domainWidePolicy" : multi_bytes,
	"drink" : multi_str,
	"driverName" : single_str,
	"driverVersion" : single_int,
	"dSCorePropagationData" : multi_date,
	"dSHeuristics" : single_str,
	"dSUIAdminMaximum" : single_int,
	"dSUIAdminNotification" : multi_str,
	"dSUIShellMaximum" : single_int,
	"dSASignature" : single_bytes,
	"dynamicLDAPServer" : single_str,
	"mail" : single_str,
	"eFSPolicy" : multi_bytes,
	"employeeID" : single_str,
	"employeeNumber" : single_str,
	"employeeType" : single_str,
	"Enabled" : single_bool,
	"enabledConnection" : single_bool,
	"enrollmentProviders" : single_str,
	"entryTTL" : single_int,
	"extendedAttributeInfo" : multi_str,
	"extendedCharsAllowed" : single_bool,
	"extendedClassInfo" : multi_str,
	"extensionName" : multi_str,
	"extraColumns" : multi_str,
	"facsimileTelephoneNumber" : single_str,
	"fileExtPriority" : multi_str,
	"flags" : single_int,
	"flatName" : single_str,
	"forceLogoff" : single_int,
	"foreignIdentifier" : single_bytes,
	"friendlyNames" : multi_str,
	"fromEntry" : multi_bool,
	"fromServer" : single_str,
	"frsComputerReference" : single_str,
	"frsComputerReferenceBL" : multi_str,
	"fRSControlDataCreation" : single_str,
	"fRSControlInboundBacklog" : single_str,
	"fRSControlOutboundBacklog" : single_str,
	"fRSDirectoryFilter" : single_str,
	"fRSDSPoll" : single_int,
	"fRSExtensions" : single_bytes,
	"fRSFaultCondition" : single_str,
	"fRSFileFilter" : single_str,
	"fRSFlags" : single_int,
	"fRSLevelLimit" : single_int,
	"fRSMemberReference" : single_str,
	"fRSMemberReferenceBL" : multi_str,
	"fRSPartnerAuthLevel" : single_int,
	"fRSPrimaryMember" : single_str,
	"fRSReplicaSetGUID" : single_bytes,
	"fRSReplicaSetType" : single_int,
	"fRSRootPath" : single_str,
	"fRSRootSecurity" : single_sd,
	"fRSServiceCommand" : single_str,
	"fRSServiceCommandStatus" : single_str,
	"fRSStagingPath" : single_str,
	"fRSTimeLastCommand" : single_date,
	"fRSTimeLastConfigChange" : single_date,
	"fRSUpdateTimeout" : single_int,
	"fRSVersion" : single_str,
	"fRSVersionGUID" : single_bytes,
	"fRSWorkingPath" : single_str,
	"fSMORoleOwner" : single_str,
	"garbageCollPeriod" : single_int,
	"gecos" : single_str,
	"generatedConnection" : single_bool,
	"generationQualifier" : single_str,
	"gidNumber" : single_int,
	"givenName" : single_str,
	"globalAddressList" : multi_str,
	"globalAddressList2" : multi_str,
	"governsID" : single_str,
	"gPLink" : single_str,
	"gPOptions" : single_int,
	"gPCFileSysPath" : single_str,
	"gPCFunctionalityVersion" : single_int,
	"gPCMachineExtensionNames" : single_str,
	"gPCUserExtensionNames" : single_str,
	"gPCWQLFilter" : single_str,
	"groupAttributes" : single_int,
	"groupMembershipSAM" : single_bytes,
	"groupPriority" : multi_str,
	"groupType" : single_int,
	"groupsToIgnore" : multi_str,
	"hasMasterNCs" : multi_str,
	"hasPartialReplicaNCs" : multi_str,
	"helpData16" : single_bytes,
	"helpData32" : single_bytes,
	"helpFileName" : single_str,
	"hideFromAB" : single_bool,
	"homeDirectory" : single_str,
	"homeDrive" : single_str,
	"host" : multi_str,
	"houseIdentifier" : multi_str,
	"iconPath" : multi_str,
	"implementedCategories" : multi_bytes,
	"indexedScopes" : multi_str,
	"initialAuthIncoming" : single_str,
	"initialAuthOutgoing" : single_str,
	"initials" : single_str,
	"installUiLevel" : single_int,
	"instanceType" : single_int,
	"interSiteTopologyFailover" : single_int,
	"interSiteTopologyGenerator" : single_str,
	"interSiteTopologyRenew" : single_int,
	"internationalISDNNumber" : multi_str,
	"invocationId" : single_bytes,
	"ipHostNumber" : multi_str,
	"ipNetmaskNumber" : single_str,
	"ipNetworkNumber" : single_str,
	"ipProtocolNumber" : single_int,
	"ipsecData" : single_bytes,
	"ipsecDataType" : single_int,
	"ipsecFilterReference" : multi_str,
	"ipsecID" : single_str,
	"ipsecISAKMPReference" : single_str,
	"ipsecName" : single_str,
	"iPSECNegotiationPolicyAction" : single_str,
	"ipsecNegotiationPolicyReference" : single_str,
	"iPSECNegotiationPolicyType" : single_str,
	"ipsecNFAReference" : multi_str,
	"ipsecOwnersReference" : multi_str,
	"ipsecPolicyReference" : single_str,
	"ipServicePort" : single_int,
	"ipServiceProtocol" : multi_str,
	"isCriticalSystemObject" : single_bool,
	"isDefunct" : single_bool,
	"isDeleted" : single_bool,
	"isEphemeral" : single_bool,
	"memberOf" : multi_str,
	"isMemberOfPartialAttributeSet" : single_bool,
	"isPrivilegeHolder" : multi_str,
	"isRecycled" : single_bool,
	"isSingleValued" : single_bool,
	"jpegPhoto" : multi_bytes,
	"keywords" : multi_str,
	"knowledgeInformation" : multi_str,
	"labeledURI" : multi_str,
	"lastBackupRestorationTime" : single_int,
	"lastContentIndexed" : single_int,
	"lastKnownParent" : single_str,
	"lastLogoff" : single_int,
	"lastLogon" : single_int,
	"lastLogonTimestamp" : single_int,
	"lastSetTime" : single_int,
	"lastUpdateSequence" : single_str,
	"lDAPAdminLimits" : multi_str,
	"lDAPDisplayName" : single_str,
	"lDAPIPDenyList" : multi_bytes,
	"legacyExchangeDN" : single_str,
	"linkID" : single_int,
	"linkTrackSecret" : single_bytes,
	"lmPwdHistory" : multi_bytes,
	"localPolicyFlags" : single_int,
	"localPolicyReference" : single_str,
	"localeID" : multi_int,
	"l" : single_str,
	"localizationDisplayId" : single_int,
	"localizedDescription" : multi_str,
	"location" : single_str,
	"lockOutObservationWindow" : single_int,
	"lockoutDuration" : single_int,
	"lockoutThreshold" : single_int,
	"lockoutTime" : single_int,
	"loginShell" : single_str,
	"thumbnailLogo" : single_bytes,
	"logonCount" : single_int,
	"logonHours" : single_bytes,
	"logonWorkstation" : single_bytes,
	"lSACreationTime" : single_int,
	"lSAModifiedCount" : single_int,
	"macAddress" : multi_str,
	"machineArchitecture" : multi_int,
	"machinePasswordChangeInterval" : single_int,
	"machineRole" : single_int,
	"machineWidePolicy" : multi_bytes,
	"managedBy" : single_str,
	"managedObjects" : multi_str,
	"manager" : single_str,
	"mAPIID" : single_int,
	"marshalledInterface" : multi_bytes,
	"masteredBy" : multi_str,
	"maxPwdAge" : single_int,
	"maxRenewAge" : single_int,
	"maxStorage" : single_int,
	"maxTicketAge" : single_int,
	"mayContain" : multi_str,
	"meetingAdvertiseScope" : single_str,
	"meetingApplication" : multi_str,
	"meetingBandwidth" : multi_int,
	"meetingBlob" : single_bytes,
	"meetingContactInfo" : single_str,
	"meetingDescription" : single_str,
	"meetingEndTime" : multi_date,
	"meetingID" : single_str,
	"meetingIP" : single_str,
	"meetingIsEncrypted" : single_str,
	"meetingKeyword" : multi_str,
	"meetingLanguage" : multi_str,
	"meetingLocation" : multi_str,
	"meetingMaxParticipants" : single_int,
	"meetingName" : single_str,
	"meetingOriginator" : single_str,
	"meetingOwner" : single_str,
	"meetingProtocol" : multi_str,
	"meetingRating" : multi_str,
	"meetingRecurrence" : single_str,
	"meetingScope" : multi_str,
	"meetingStartTime" : multi_date,
	"meetingType" : single_str,
	"meetingURL" : multi_str,
	"member" : multi_str,
	"memberNisNetgroup" : multi_str,
	"memberUid" : multi_str,
	"mhsORAddress" : multi_str,
	"minPwdAge" : single_int,
	"minPwdLength" : single_int,
	"minTicketAge" : single_int,
	"modifiedCount" : single_int,
	"modifiedCountAtLastProm" : single_int,
	"modifyTimeStamp" : single_date,
	"moniker" : multi_bytes,
	"monikerDisplayName" : multi_str,
	"moveTreeState" : multi_bytes,
	"msAuthz-CentralAccessPolicyID" : single_sid,
	"msAuthz-EffectiveSecurityPolicy" : single_str,
	"msAuthz-LastEffectiveSecurityPolicy" : single_str,
	"msAuthz-MemberRulesInCentralAccessPolicy" : multi_str,
	"msAuthz-MemberRulesInCentralAccessPolicyBL" : multi_str,
	"msAuthz-ProposedSecurityPolicy" : single_str,
	"msAuthz-ResourceCondition" : single_str,
	"msCOM-DefaultPartitionLink" : single_str,
	"msCOM-ObjectId" : single_bytes,
	"msCOM-PartitionLink" : multi_str,
	"msCOM-PartitionSetLink" : multi_str,
	"msCOM-UserLink" : multi_str,
	"msCOM-UserPartitionSetLink" : single_str,
	"msDFS-Commentv2" : single_str,
	"msDFS-GenerationGUIDv2" : single_bytes,
	"msDFS-LastModifiedv2" : single_date,
	"msDFS-LinkIdentityGUIDv2" : single_bytes,
	"msDFS-LinkPathv2" : single_str,
	"msDFS-LinkSecurityDescriptorv2" : single_sd,
	"msDFS-NamespaceIdentityGUIDv2" : single_bytes,
	"msDFS-Propertiesv2" : multi_str,
	"msDFS-SchemaMajorVersion" : single_int,
	"msDFS-SchemaMinorVersion" : single_int,
	"msDFS-ShortNameLinkPathv2" : single_str,
	"msDFS-TargetListv2" : single_bytes,
	"msDFS-Ttlv2" : single_int,
	"msDFSR-CachePolicy" : single_int,
	"msDFSR-CommonStagingPath" : single_str,
	"msDFSR-CommonStagingSizeInMb" : single_int,
	"msDFSR-ComputerReference" : single_str,
	"msDFSR-ComputerReferenceBL" : multi_str,
	"msDFSR-ConflictPath" : single_str,
	"msDFSR-ConflictSizeInMb" : single_int,
	"msDFSR-ContentSetGuid" : single_bytes,
	"msDFSR-DefaultCompressionExclusionFilter" : single_str,
	"msDFSR-DeletedPath" : single_str,
	"msDFSR-DeletedSizeInMb" : single_int,
	"msDFSR-DfsLinkTarget" : single_str,
	"msDFSR-DfsPath" : single_str,
	"msDFSR-DirectoryFilter" : single_str,
	"msDFSR-DisablePacketPrivacy" : single_bool,
	"msDFSR-Enabled" : single_bool,
	"msDFSR-Extension" : single_bytes,
	"msDFSR-FileFilter" : single_str,
	"msDFSR-Flags" : single_int,
	"msDFSR-Keywords" : single_str,
	"msDFSR-MaxAgeInCacheInMin" : single_int,
	"msDFSR-MemberReference" : single_str,
	"msDFSR-MemberReferenceBL" : multi_str,
	"msDFSR-MinDurationCacheInMin" : single_int,
	"msDFSR-OnDemandExclusionDirectoryFilter" : single_str,
	"msDFSR-OnDemandExclusionFileFilter" : single_str,
	"msDFSR-Options" : single_int,
	"msDFSR-Options2" : single_int,
	"msDFSR-Priority" : single_int,
	"msDFSR-RdcEnabled" : single_bool,
	"msDFSR-RdcMinFileSizeInKb" : single_int,
	"msDFSR-ReadOnly" : single_bool,
	"msDFSR-ReplicationGroupGuid" : single_bytes,
	"msDFSR-ReplicationGroupType" : single_int,
	"msDFSR-RootFence" : single_int,
	"msDFSR-RootPath" : single_str,
	"msDFSR-RootSizeInMb" : single_int,
	"msDFSR-Schedule" : single_bytes,
	"msDFSR-StagingCleanupTriggerInPercent" : single_int,
	"msDFSR-StagingPath" : single_str,
	"msDFSR-StagingSizeInMb" : single_int,
	"msDFSR-TombstoneExpiryInMin" : single_int,
	"msDFSR-Version" : single_str,
	"msDNS-DNSKEYRecordSetTTL" : single_int,
	"msDNS-DNSKEYRecords" : multi_bytes,
	"msDNS-DSRecordAlgorithms" : single_int,
	"msDNS-DSRecordSetTTL" : single_int,
	"msDNS-IsSigned" : single_bool,
	"msDNS-KeymasterZones" : multi_str,
	"msDNS-MaintainTrustAnchor" : single_int,
	"msDNS-NSEC3CurrentSalt" : single_str,
	"msDNS-NSEC3HashAlgorithm" : single_int,
	"msDNS-NSEC3Iterations" : single_int,
	"msDNS-NSEC3OptOut" : single_bool,
	"msDNS-NSEC3RandomSaltLength" : single_int,
	"msDNS-NSEC3UserSalt" : single_str,
	"msDNS-ParentHasSecureDelegation" : single_bool,
	"msDNS-PropagationTime" : single_int,
	"msDNS-RFC5011KeyRollovers" : single_bool,
	"msDNS-SecureDelegationPollingPeriod" : single_int,
	"msDNS-SignWithNSEC3" : single_bool,
	"msDNS-SignatureInceptionOffset" : single_int,
	"msDNS-SigningKeyDescriptors" : multi_bytes,
	"msDNS-SigningKeys" : multi_bytes,
	"msDRM-IdentityCertificate" : multi_bytes,
	"msDS-AdditionalDnsHostName" : multi_str,
	"msDS-AdditionalSamAccountName" : multi_str,
	"msDS-AllUsersTrustQuota" : single_int,
	"msDS-AllowedDNSSuffixes" : multi_str,
	"msDS-AllowedToActOnBehalfOfOtherIdentity" : single_sd,
	"msDS-AllowedToDelegateTo" : multi_str,
	"msDS-AppliesToResourceTypes" : multi_str,
	"msDS-Approx-Immed-Subordinates" : single_int,
	"msDS-ApproximateLastLogonTimeStamp" : single_int,
	"msDS-AssignedAuthNPolicy" : single_str,
	"msDS-AssignedAuthNPolicyBL" : multi_str,
	"msDS-AssignedAuthNPolicySilo" : single_str,
	"msDS-AssignedAuthNPolicySiloBL" : multi_str,
	"msDS-AuthenticatedAtDC" : multi_str,
	"msDS-AuthenticatedToAccountlist" : multi_str,
	"msDS-AuthNPolicyEnforced" : single_bool,
	"msDS-AuthNPolicySiloEnforced" : single_bool,
	"msDS-AuthNPolicySiloMembers" : multi_str,
	"msDS-AuthNPolicySiloMembersBL" : multi_str,
	"msDS-Auxiliary-Classes" : multi_str,
	"msDS-AzApplicationData" : single_str,
	"msDS-AzApplicationName" : single_str,
	"msDS-AzApplicationVersion" : single_str,
	"msDS-AzBizRule" : single_str,
	"msDS-AzBizRuleLanguage" : single_str,
	"msDS-AzClassId" : single_str,
	"msDS-AzDomainTimeout" : single_int,
	"msDS-AzGenerateAudits" : single_bool,
	"msDS-AzGenericData" : single_str,
	"msDS-AzLastImportedBizRulePath" : single_str,
	"msDS-AzLDAPQuery" : single_str,
	"msDS-AzMajorVersion" : single_int,
	"msDS-AzMinorVersion" : single_int,
	"msDS-AzObjectGuid" : single_bytes,
	"msDS-AzOperationID" : single_int,
	"msDS-AzScopeName" : single_str,
	"msDS-AzScriptEngineCacheMax" : single_int,
	"msDS-AzScriptTimeout" : single_int,
	"msDS-AzTaskIsRoleDefinition" : single_bool,
	"msDS-Behavior-Version" : single_int,
	"msDS-BridgeHeadServersUsed" : multi_str,
	"msDS-ByteArray" : multi_bytes,
	"msDS-Cached-Membership" : single_bytes,
	"msDS-Cached-Membership-Time-Stamp" : single_int,
	"msDS-ClaimAttributeSource" : single_str,
	"msDS-ClaimIsSingleValued" : single_bool,
	"msDS-ClaimIsValueSpaceRestricted" : single_bool,
	"msDS-ClaimPossibleValues" : single_str,
	"msDS-ClaimSharesPossibleValuesWith" : single_str,
	"msDS-ClaimSharesPossibleValuesWithBL" : multi_str,
	"msDS-ClaimSource" : single_str,
	"msDS-ClaimSourceType" : single_str,
	"msDS-ClaimTypeAppliesToClass" : multi_str,
	"msDS-ClaimValueType" : single_int,
	"msDS-CloudAnchor" : single_bytes,
	"msDS-CloudIsEnabled" : single_bool,
	"msDS-CloudIsManaged" : single_bool,
	"msDS-CloudIssuerPublicCertificates" : multi_bytes,
	"msDS-cloudExtensionAttribute1" : single_str,
	"msDS-cloudExtensionAttribute10" : single_str,
	"msDS-cloudExtensionAttribute11" : single_str,
	"msDS-cloudExtensionAttribute12" : single_str,
	"msDS-cloudExtensionAttribute13" : single_str,
	"msDS-cloudExtensionAttribute14" : single_str,
	"msDS-cloudExtensionAttribute15" : single_str,
	"msDS-cloudExtensionAttribute16" : single_str,
	"msDS-cloudExtensionAttribute17" : single_str,
	"msDS-cloudExtensionAttribute18" : single_str,
	"msDS-cloudExtensionAttribute19" : single_str,
	"msDS-cloudExtensionAttribute2" : single_str,
	"msDS-cloudExtensionAttribute20" : single_str,
	"msDS-cloudExtensionAttribute3" : single_str,
	"msDS-cloudExtensionAttribute4" : single_str,
	"msDS-cloudExtensionAttribute5" : single_str,
	"msDS-cloudExtensionAttribute6" : single_str,
	"msDS-cloudExtensionAttribute7" : single_str,
	"msDS-cloudExtensionAttribute8" : single_str,
	"msDS-cloudExtensionAttribute9" : single_str,
	"msDS-ComputerAllowedToAuthenticateTo" : single_bytes,
	"msDS-ComputerAuthNPolicy" : single_str,
	"msDS-ComputerAuthNPolicyBL" : multi_str,
	"msDS-ComputerSID" : single_sid,
	"msDS-ComputerTGTLifetime" : single_int,
	"mS-DS-ConsistencyChildCount" : single_int,
	"mS-DS-ConsistencyGuid" : single_bytes,
	"mS-DS-CreatorSID" : single_sid,
	"msDS-CustomKeyInformation" : single_bytes,
	"msDS-DateTime" : multi_date,
	"msDS-DefaultQuota" : single_int,
	"msDS-DeletedObjectLifetime" : single_int,
	"msDS-DeviceDN" : single_str,
	"msDS-DeviceID" : single_bytes,
	"msDS-DeviceLocation" : single_str,
	"msDS-DeviceMDMStatus" : single_str,
	"msDS-DeviceObjectVersion" : single_int,
	"msDS-DeviceOSType" : single_str,
	"msDS-DeviceOSVersion" : single_str,
	"msDS-DevicePhysicalIDs" : multi_str,
	"msDS-DeviceTrustType" : single_int,
	"msDS-DnsRootAlias" : single_str,
	"msDS-DrsFarmID" : single_str,
	"msDS-EgressClaimsTransformationPolicy" : single_str,
	"msDS-EnabledFeature" : multi_str,
	"msDS-EnabledFeatureBL" : multi_str,
	"msDS-Entry-Time-To-Die" : single_date,
	"msDS-ExecuteScriptPassword" : single_bytes,
	"msDS-ExpirePasswordsOnSmartCardOnlyAccounts" : single_bool,
	"msDS-ExternalDirectoryObjectId" : single_str,
	"msDS-ExternalKey" : multi_str,
	"msDS-ExternalStore" : multi_str,
	"msDS-FailedInteractiveLogonCount" : single_int,
	"msDS-FailedInteractiveLogonCountAtLastSuccessfulLogon" : single_int,
	"msDS-FilterContainers" : multi_str,
	"msDS-GenerationId" : single_bytes,
	"msDS-GeoCoordinatesAltitude" : single_int,
	"msDS-GeoCoordinatesLatitude" : single_int,
	"msDS-GeoCoordinatesLongitude" : single_int,
	"msDS-GroupMSAMembership" : single_sd,
	"msDS-HABSeniorityIndex" : single_int,
	"msDS-HasDomainNCs" : multi_str,
	"msDS-hasFullReplicaNCs" : multi_str,
	"msDS-HasInstantiatedNCs" : multi_str,
	"msDS-hasMasterNCs" : multi_str,
	"msDS-HostServiceAccount" : multi_str,
	"msDS-HostServiceAccountBL" : multi_str,
	"msDS-IngressClaimsTransformationPolicy" : single_str,
	"msDS-Integer" : multi_int,
	"msDS-IntId" : single_int,
	"msDS-IsCompliant" : single_bool,
	"msDS-IsDomainFor" : multi_str,
	"msDS-IsEnabled" : single_bool,
	"msDS-IsFullReplicaFor" : multi_str,
	"msds-memberOfTransitive" : multi_str,
	"msDS-IsPartialReplicaFor" : multi_str,
	"msDS-IsPossibleValuesPresent" : single_bool,
	"msDS-IsPrimaryComputerFor" : multi_str,
	"msDS-IsUsedAsResourceSecurityAttribute" : single_bool,
	"msDS-IsUserCachableAtRodc" : single_int,
	"msDS-isGC" : single_bool,
	"msDS-IsManaged" : single_bool,
	"msDS-isRODC" : single_bool,
	"msDS-IssuerCertificates" : multi_bytes,
	"msDS-IssuerPublicCertificates" : multi_bytes,
	"msDS-KeyApproximateLastLogonTimeStamp" : single_int,
	"msDS-KeyCredentialLink" : multi_str,
	"msDS-KeyCredentialLink-BL" : multi_str,
	"msDS-KeyId" : single_bytes,
	"msDS-KeyMaterial" : single_bytes,
	"msDS-KeyPrincipal" : single_str,
	"msDS-KeyPrincipalBL" : multi_str,
	"msDS-KeyUsage" : single_str,
	"msDS-KeyVersionNumber" : single_int,
	"msDS-KrbTgtLink" : single_str,
	"msDS-KrbTgtLinkBl" : multi_str,
	"msDS-LastFailedInteractiveLogonTime" : single_int,
	"msDS-LastKnownRDN" : single_str,
	"msDS-LastSuccessfulInteractiveLogonTime" : single_int,
	"msDS-LocalEffectiveDeletionTime" : single_date,
	"msDS-LocalEffectiveRecycleTime" : single_date,
	"msDS-LockoutDuration" : single_int,
	"msDS-LockoutObservationWindow" : single_int,
	"msDS-LockoutThreshold" : single_int,
	"msDS-LogonTimeSyncInterval" : single_int,
	"ms-DS-MachineAccountQuota" : single_int,
	"msDS-ManagedPassword" : single_bytes,
	"msDS-ManagedPasswordId" : single_bytes,
	"msDS-ManagedPasswordInterval" : single_int,
	"msDS-ManagedPasswordPreviousId" : single_bytes,
	"msDs-masteredBy" : multi_str,
	"msDs-MaxValues" : single_int,
	"msDS-MaximumPasswordAge" : single_int,
	"msDS-MaximumRegistrationInactivityPeriod" : single_int,
	"msds-memberTransitive" : multi_str,
	"msDS-MembersForAzRole" : multi_str,
	"msDS-MembersForAzRoleBL" : multi_str,
	"msDS-MembersOfResourcePropertyList" : multi_str,
	"msDS-MembersOfResourcePropertyListBL" : multi_str,
	"msDS-MinimumPasswordAge" : single_int,
	"msDS-MinimumPasswordLength" : single_int,
	"msDS-NCReplCursors" : multi_str,
	"msDS-NCReplInboundNeighbors" : multi_str,
	"msDS-NCReplOutboundNeighbors" : multi_str,
	"msDS-NC-Replica-Locations" : multi_str,
	"msDS-NC-RO-Replica-Locations" : multi_str,
	"msDS-NC-RO-Replica-Locations-BL" : multi_str,
	"msDS-NcType" : single_int,
	"msDS-NeverRevealGroup" : multi_str,
	"msDS-NonMembers" : multi_str,
	"msDS-NonMembersBL" : multi_str,
	"msDS-Non-Security-Group-Extra-Classes" : multi_str,
	"msDS-ObjectReference" : multi_str,
	"msDS-ObjectReferenceBL" : multi_str,
	"msDS-ObjectSoa" : single_str,
	"msDS-OIDToGroupLink" : single_str,
	"msDS-OIDToGroupLinkBl" : multi_str,
	"msDS-OperationsForAzRole" : multi_str,
	"msDS-OperationsForAzRoleBL" : multi_str,
	"msDS-OperationsForAzTask" : multi_str,
	"msDS-OperationsForAzTaskBL" : multi_str,
	"msDS-OptionalFeatureFlags" : single_int,
	"msDS-OptionalFeatureGUID" : single_bytes,
	"msDS-Other-Settings" : multi_str,
	"msDS-parentdistname" : single_str,
	"msDS-PasswordComplexityEnabled" : single_bool,
	"msDS-PasswordHistoryLength" : single_int,
	"msDS-PasswordReversibleEncryptionEnabled" : single_bool,
	"msDS-PasswordSettingsPrecedence" : single_int,
	"msDS-PerUserTrustQuota" : single_int,
	"msDS-PerUserTrustTombstonesQuota" : single_int,
	"msDS-PhoneticCompanyName" : single_str,
	"msDS-PhoneticDepartment" : single_str,
	"msDS-PhoneticDisplayName" : single_str,
	"msDS-PhoneticFirstName" : single_str,
	"msDS-PhoneticLastName" : single_str,
	"msDS-preferredDataLocation" : single_str,
	"msDS-Preferred-GC-Site" : single_str,
	"msDS-PrimaryComputer" : multi_str,
	"msDS-PrincipalName" : single_str,
	"msDS-PromotionSettings" : single_str,
	"msDS-PSOApplied" : multi_str,
	"msDS-PSOAppliesTo" : multi_str,
	"msDS-QuotaAmount" : single_int,
	"msDS-QuotaEffective" : single_int,
	"msDS-QuotaTrustee" : single_sid,
	"msDS-QuotaUsed" : single_int,
	"msDS-RegisteredOwner" : single_bytes,
	"msDS-RegisteredUsers" : multi_bytes,
	"msDS-RegistrationQuota" : single_int,
	"msDS-ReplAttributeMetaData" : multi_str,
	"msDS-ReplValueMetaData" : multi_str,
	"msDS-ReplValueMetaDataExt" : multi_str,
	"mS-DS-ReplicatesNCReason" : multi_str,
	"msDS-Replication-Notify-First-DSA-Delay" : single_int,
	"msDS-Replication-Notify-Subsequent-DSA-Delay" : single_int,
	"msDS-ReplicationEpoch" : single_int,
	"msDS-RequiredDomainBehaviorVersion" : single_int,
	"msDS-RequiredForestBehaviorVersion" : single_int,
	"msDS-ResultantPSO" : single_str,
	"msDS-RetiredReplNCSignatures" : single_bytes,
	"msDS-RevealOnDemandGroup" : multi_str,
	"msDS-RevealedDSAs" : multi_str,
	"msDS-RevealedList" : multi_str,
	"msDS-RevealedListBL" : multi_str,
	"msDS-RevealedUsers" : multi_str,
	"msDS-RIDPoolAllocationEnabled" : single_bool,
	"msDs-Schema-Extensions" : multi_bytes,
	"msDS-SDReferenceDomain" : single_str,
	"msDS-SecondaryKrbTgtNumber" : single_int,
	"msDS-Security-Group-Extra-Classes" : multi_str,
	"msDS-ServiceAllowedNTLMNetworkAuthentication" : single_bool,
	"msDS-ServiceAllowedToAuthenticateFrom" : single_bytes,
	"msDS-ServiceAllowedToAuthenticateTo" : single_bytes,
	"msDS-ServiceAuthNPolicy" : single_str,
	"msDS-ServiceAuthNPolicyBL" : multi_str,
	"msDS-ServiceTGTLifetime" : single_int,
	"msDS-Settings" : multi_str,
	"msDS-ShadowPrincipalSid" : single_sid,
	"msDS-Site-Affinity" : multi_bytes,
	"msDS-SiteName" : single_str,
	"msDS-SourceAnchor" : single_str,
	"msDS-SourceObjectDN" : single_str,
	"msDS-SPNSuffixes" : multi_str,
	"msDS-StrongNTLMPolicy" : single_int,
	"msDS-SupportedEncryptionTypes" : single_int,
	"msDS-SyncServerUrl" : multi_str,
	"msDS-TasksForAzRole" : multi_str,
	"msDS-TasksForAzRoleBL" : multi_str,
	"msDS-TasksForAzTask" : multi_str,
	"msDS-TasksForAzTaskBL" : multi_str,
	"msDS-TDOEgressBL" : multi_str,
	"msDS-TDOIngressBL" : multi_str,
	"msds-tokenGroupNames" : multi_str,
	"msds-tokenGroupNamesGlobalAndUniversal" : multi_str,
	"msds-tokenGroupNamesNoGCAcceptable" : multi_str,
	"msDS-TombstoneQuotaFactor" : single_int,
	"msDS-TopQuotaUsage" : multi_str,
	"msDS-TransformationRules" : single_str,
	"msDS-TransformationRulesCompiled" : single_bytes,
	"msDS-TrustForestTrustInfo" : single_bytes,
	"msDS-UpdateScript" : single_str,
	"msDS-User-Account-Control-Computed" : single_int,
	"msDS-UserAllowedNTLMNetworkAuthentication" : single_bool,
	"msDS-UserAllowedToAuthenticateFrom" : single_bytes,
	"msDS-UserAllowedToAuthenticateTo" : single_bytes,
	"msDS-UserAuthNPolicy" : single_str,
	"msDS-UserAuthNPolicyBL" : multi_str,
	"msDS-UserPasswordExpiryTimeComputed" : single_int,
	"msDS-UserTGTLifetime" : single_int,
	"msDS-USNLastSyncSuccess" : single_int,
	"msDS-ValueTypeReference" : single_str,
	"msDS-ValueTypeReferenceBL" : multi_str,
	"msExchAssistantName" : single_str,
	"msExchHouseIdentifier" : single_str,
	"msExchLabeledURI" : multi_str,
	"ownerBL" : multi_str,
	"msFRS-Hub-Member" : single_str,
	"msFRS-Topology-Pref" : single_str,
	"msFVE-KeyPackage" : single_bytes,
	"msFVE-RecoveryGuid" : single_bytes,
	"msFVE-RecoveryPassword" : single_str,
	"msFVE-VolumeGuid" : single_bytes,
	"msieee80211-Data" : single_bytes,
	"msieee80211-DataType" : single_int,
	"msieee80211-ID" : single_str,
	"msIIS-FTPDir" : single_str,
	"msIIS-FTPRoot" : single_str,
	"msImaging-HashAlgorithm" : single_str,
	"msImaging-PSPIdentifier" : single_bytes,
	"msImaging-PSPString" : single_str,
	"msImaging-ThumbprintHash" : single_bytes,
	"msKds-CreateTime" : single_int,
	"msKds-DomainID" : single_str,
	"msKds-KDFAlgorithmID" : single_str,
	"msKds-KDFParam" : single_bytes,
	"msKds-PrivateKeyLength" : single_int,
	"msKds-PublicKeyLength" : single_int,
	"msKds-RootKeyData" : single_bytes,
	"msKds-SecretAgreementAlgorithmID" : single_str,
	"msKds-SecretAgreementParam" : single_bytes,
	"msKds-UseStartTime" : single_int,
	"msKds-Version" : single_int,
	"ms-Mcs-AdmPwd" : single_str,
	"ms-Mcs-AdmPwdExpirationTime" : single_int,
	"ms-net-ieee-80211-GP-PolicyData" : single_str,
	"ms-net-ieee-80211-GP-PolicyGUID" : single_str,
	"ms-net-ieee-80211-GP-PolicyReserved" : single_bytes,
	"ms-net-ieee-8023-GP-PolicyData" : single_str,
	"ms-net-ieee-8023-GP-PolicyGUID" : single_str,
	"ms-net-ieee-8023-GP-PolicyReserved" : single_bytes,
	"msPKIAccountCredentials" : multi_str,
	"msPKI-Cert-Template-OID" : single_str,
	"msPKI-Certificate-Application-Policy" : multi_str,
	"msPKI-Certificate-Name-Flag" : single_int,
	"msPKI-Certificate-Policy" : multi_str,
	"msPKI-CredentialRoamingTokens" : multi_str,
	"msPKIDPAPIMasterKeys" : multi_str,
	"msPKI-Enrollment-Flag" : single_int,
	"msPKI-Enrollment-Servers" : multi_str,
	"msPKI-Minimal-Key-Size" : single_int,
	"msPKI-OID-Attribute" : single_int,
	"msPKI-OID-CPS" : multi_str,
	"msPKI-OIDLocalizedName" : multi_str,
	"msPKI-OID-User-Notice" : multi_str,
	"msPKI-Private-Key-Flag" : single_int,
	"msPKI-RA-Application-Policies" : multi_str,
	"msPKI-RA-Policies" : multi_str,
	"msPKI-RA-Signature" : single_int,
	"msPKIRoamingTimeStamp" : single_bytes,
	"msPKI-Site-Name" : single_str,
	"msPKI-Supersede-Templates" : multi_str,
	"msPKI-Template-Minor-Revision" : single_int,
	"msPKI-Template-Schema-Version" : single_int,
	"msRADIUS-FramedInterfaceId" : single_str,
	"msRADIUS-FramedIpv6Prefix" : single_str,
	"msRADIUS-FramedIpv6Route" : multi_str,
	"msRADIUS-SavedFramedInterfaceId" : single_str,
	"msRADIUS-SavedFramedIpv6Prefix" : single_str,
	"msRADIUS-SavedFramedIpv6Route" : multi_str,
	"msRRASAttribute" : multi_str,
	"msRRASVendorAttributeEntry" : multi_str,
	"msSPP-ConfigLicense" : single_bytes,
	"msSPP-ConfirmationId" : single_str,
	"msSPP-CSVLKPartialProductKey" : single_str,
	"msSPP-CSVLKPid" : single_str,
	"msSPP-CSVLKSkuId" : single_bytes,
	"msSPP-InstallationId" : single_str,
	"msSPP-IssuanceLicense" : single_bytes,
	"msSPP-KMSIds" : multi_bytes,
	"msSPP-OnlineLicense" : single_bytes,
	"msSPP-PhoneLicense" : single_bytes,
	"mS-SQL-Alias" : single_str,
	"mS-SQL-AllowAnonymousSubscription" : single_bool,
	"mS-SQL-AllowImmediateUpdatingSubscription" : single_bool,
	"mS-SQL-AllowKnownPullSubscription" : single_bool,
	"mS-SQL-AllowQueuedUpdatingSubscription" : single_bool,
	"mS-SQL-AllowSnapshotFilesFTPDownloading" : single_bool,
	"mS-SQL-AppleTalk" : single_str,
	"mS-SQL-Applications" : multi_str,
	"mS-SQL-Build" : single_int,
	"mS-SQL-CharacterSet" : single_int,
	"mS-SQL-Clustered" : single_bool,
	"mS-SQL-ConnectionURL" : single_str,
	"mS-SQL-Contact" : single_str,
	"mS-SQL-CreationDate" : single_str,
	"mS-SQL-Database" : single_str,
	"mS-SQL-Description" : single_str,
	"mS-SQL-GPSHeight" : single_str,
	"mS-SQL-GPSLatitude" : single_str,
	"mS-SQL-GPSLongitude" : single_str,
	"mS-SQL-InformationDirectory" : single_bool,
	"mS-SQL-InformationURL" : single_str,
	"mS-SQL-Keywords" : multi_str,
	"mS-SQL-Language" : single_str,
	"mS-SQL-LastBackupDate" : single_str,
	"mS-SQL-LastDiagnosticDate" : single_str,
	"mS-SQL-LastUpdatedDate" : single_str,
	"mS-SQL-Location" : single_str,
	"mS-SQL-Memory" : single_int,
	"mS-SQL-MultiProtocol" : single_str,
	"mS-SQL-Name" : single_str,
	"mS-SQL-NamedPipe" : single_str,
	"mS-SQL-PublicationURL" : single_str,
	"mS-SQL-Publisher" : single_str,
	"mS-SQL-RegisteredOwner" : single_str,
	"mS-SQL-ServiceAccount" : single_str,
	"mS-SQL-Size" : single_int,
	"mS-SQL-SortOrder" : single_str,
	"mS-SQL-SPX" : single_str,
	"mS-SQL-Status" : single_int,
	"mS-SQL-TCPIP" : single_str,
	"mS-SQL-ThirdParty" : single_bool,
	"mS-SQL-Type" : single_str,
	"mS-SQL-UnicodeSortOrder" : single_int,
	"mS-SQL-Version" : single_str,
	"mS-SQL-Vines" : single_str,
	"msTAPI-ConferenceBlob" : single_bytes,
	"msTAPI-IpAddress" : multi_str,
	"msTAPI-ProtocolId" : single_str,
	"msTAPI-uid" : single_str,
	"msTPM-OwnerInformationTemp" : single_str,
	"msTPM-OwnerInformation" : single_str,
	"msTPM-SrkPubThumbprint" : single_bytes,
	"msTPM-TpmInformationForComputer" : single_str,
	"msTPM-TpmInformationForComputerBL" : multi_str,
	"msTSAllowLogon" : single_bool,
	"msTSBrokenConnectionAction" : single_bool,
	"msTSConnectClientDrives" : single_bool,
	"msTSConnectPrinterDrives" : single_bool,
	"msTSDefaultToMainPrinter" : single_bool,
	"msTSEndpointData" : single_str,
	"msTSEndpointPlugin" : single_str,
	"msTSEndpointType" : single_int,
	"msTSExpireDate" : single_date,
	"msTSExpireDate2" : single_date,
	"msTSExpireDate3" : single_date,
	"msTSExpireDate4" : single_date,
	"msTSHomeDirectory" : single_str,
	"msTSHomeDrive" : single_str,
	"msTSInitialProgram" : single_str,
	"msTSLicenseVersion" : single_str,
	"msTSLicenseVersion2" : single_str,
	"msTSLicenseVersion3" : single_str,
	"msTSLicenseVersion4" : single_str,
	"msTSManagingLS" : single_str,
	"msTSManagingLS2" : single_str,
	"msTSManagingLS3" : single_str,
	"msTSManagingLS4" : single_str,
	"msTSMaxConnectionTime" : single_int,
	"msTSMaxDisconnectionTime" : single_int,
	"msTSMaxIdleTime" : single_int,
	"msTSPrimaryDesktop" : single_str,
	"msTSPrimaryDesktopBL" : multi_str,
	"msTSProfilePath" : single_str,
	"msTSProperty01" : multi_str,
	"msTSProperty02" : multi_str,
	"msTSReconnectionAction" : single_bool,
	"msTSRemoteControl" : single_int,
	"msTSSecondaryDesktopBL" : multi_str,
	"msTSSecondaryDesktops" : multi_str,
	"msTSWorkDirectory" : single_str,
	"msTSLSProperty01" : multi_str,
	"msTSLSProperty02" : multi_str,
	"msWMI-Author" : single_str,
	"msWMI-ChangeDate" : single_str,
	"msWMI-Class" : single_str,
	"msWMI-ClassDefinition" : single_str,
	"msWMI-CreationDate" : single_str,
	"msWMI-Genus" : single_int,
	"msWMI-ID" : single_str,
	"msWMI-Int8Default" : single_int,
	"msWMI-Int8Max" : single_int,
	"msWMI-Int8Min" : single_int,
	"msWMI-Int8ValidValues" : multi_int,
	"msWMI-IntDefault" : single_int,
	"msWMI-intFlags1" : single_int,
	"msWMI-intFlags2" : single_int,
	"msWMI-intFlags3" : single_int,
	"msWMI-intFlags4" : single_int,
	"msWMI-IntMax" : single_int,
	"msWMI-IntMin" : single_int,
	"msWMI-IntValidValues" : multi_int,
	"msWMI-Mof" : single_str,
	"msWMI-Name" : single_str,
	"msWMI-NormalizedClass" : multi_str,
	"msWMI-Parm1" : single_str,
	"msWMI-Parm2" : single_str,
	"msWMI-Parm3" : single_str,
	"msWMI-Parm4" : single_str,
	"msWMI-PropertyName" : single_str,
	"msWMI-Query" : single_str,
	"msWMI-QueryLanguage" : single_str,
	"msWMI-ScopeGuid" : single_str,
	"msWMI-SourceOrganization" : single_str,
	"msWMI-StringDefault" : single_str,
	"msWMI-StringValidValues" : multi_str,
	"msWMI-TargetClass" : single_str,
	"msWMI-TargetNameSpace" : single_str,
	"msWMI-TargetObject" : multi_bytes,
	"msWMI-TargetPath" : single_str,
	"msWMI-TargetType" : single_str,
	"mscopeId" : single_str,
	"msiFileList" : multi_str,
	"msiScript" : single_bytes,
	"msiScriptName" : single_str,
	"msiScriptPath" : single_str,
	"msiScriptSize" : single_int,
	"mSMQAuthenticate" : single_bool,
	"mSMQBasePriority" : single_int,
	"mSMQComputerType" : single_str,
	"mSMQComputerTypeEx" : single_str,
	"mSMQCost" : single_int,
	"mSMQCSPName" : single_str,
	"mSMQDependentClientService" : single_bool,
	"mSMQDependentClientServices" : single_bool,
	"mSMQDigests" : multi_bytes,
	"mSMQDigestsMig" : multi_bytes,
	"mSMQDsService" : single_bool,
	"mSMQDsServices" : single_bool,
	"mSMQEncryptKey" : single_bytes,
	"mSMQForeign" : single_bool,
	"mSMQInRoutingServers" : multi_str,
	"mSMQInterval1" : single_int,
	"mSMQInterval2" : single_int,
	"mSMQJournal" : single_bool,
	"mSMQJournalQuota" : single_int,
	"mSMQLabel" : single_str,
	"mSMQLabelEx" : single_str,
	"mSMQLongLived" : single_int,
	"mSMQMigrated" : single_bool,
	"MSMQ-MulticastAddress" : single_str,
	"mSMQNameStyle" : single_bool,
	"mSMQNt4Flags" : single_int,
	"mSMQNt4Stub" : multi_int,
	"mSMQOSType" : single_int,
	"mSMQOutRoutingServers" : multi_str,
	"mSMQOwnerID" : single_bytes,
	"mSMQPrevSiteGates" : multi_str,
	"mSMQPrivacyLevel" : single_int,
	"mSMQQMID" : single_bytes,
	"mSMQQueueJournalQuota" : single_int,
	"mSMQQueueNameExt" : single_str,
	"mSMQQueueQuota" : single_int,
	"mSMQQueueType" : single_bytes,
	"mSMQQuota" : single_int,
	"msMQ-Recipient-FormatName" : single_str,
	"mSMQRoutingService" : single_bool,
	"mSMQRoutingServices" : single_bool,
	"MSMQ-SecuredSource" : single_bool,
	"mSMQServiceType" : single_int,
	"mSMQServices" : single_int,
	"mSMQSignCertificates" : single_bytes,
	"mSMQSignCertificatesMig" : single_bytes,
	"mSMQSignKey" : single_bytes,
	"mSMQSite1" : single_str,
	"mSMQSite2" : single_str,
	"mSMQSiteForeign" : multi_bool,
	"mSMQSiteGates" : multi_str,
	"mSMQSiteGatesMig" : multi_str,
	"mSMQSiteID" : single_bytes,
	"mSMQSiteName" : single_str,
	"mSMQSiteNameEx" : single_str,
	"mSMQSites" : multi_bytes,
	"mSMQTransactional" : single_bool,
	"mSMQUserSid" : single_bytes,
	"mSMQVersion" : single_int,
	"msNPAllowDialin" : single_bool,
	"msNPCalledStationID" : multi_str,
	"msNPCallingStationID" : multi_str,
	"msNPSavedCallingStationID" : multi_str,
	"msRADIUSCallbackNumber" : single_str,
	"msRADIUSFramedIPAddress" : single_int,
	"msRADIUSFramedRoute" : multi_str,
	"msRADIUSServiceType" : single_int,
	"msRASSavedCallbackNumber" : single_str,
	"msRASSavedFramedIPAddress" : single_int,
	"msRASSavedFramedRoute" : multi_str,
	"msSFU30Aliases" : multi_str,
	"msSFU30CryptMethod" : single_str,
	"msSFU30Domains" : multi_str,
	"msSFU30FieldSeparator" : single_str,
	"msSFU30IntraFieldSeparator" : single_str,
	"msSFU30IsValidContainer" : single_int,
	"msSFU30KeyAttributes" : multi_str,
	"msSFU30KeyValues" : multi_str,
	"msSFU30MapFilter" : single_str,
	"msSFU30MasterServerName" : single_str,
	"msSFU30MaxGidNumber" : single_int,
	"msSFU30MaxUidNumber" : single_int,
	"msSFU30Name" : single_str,
	"msSFU30NetgroupHostAtDomain" : multi_str,
	"msSFU30NetgroupUserAtDomain" : multi_str,
	"msSFU30NisDomain" : single_str,
	"msSFU30NSMAPFieldPosition" : single_str,
	"msSFU30OrderNumber" : single_str,
	"msSFU30PosixMember" : multi_str,
	"msSFU30PosixMemberOf" : multi_str,
	"msSFU30ResultAttributes" : multi_str,
	"msSFU30SearchAttributes" : multi_str,
	"msSFU30SearchContainer" : single_str,
	"msSFU30YpServers" : multi_str,
	"mustContain" : multi_str,
	"nameServiceFlags" : single_int,
	"nCName" : single_str,
	"nETBIOSName" : single_str,
	"netbootAllowNewClients" : single_bool,
	"netbootAnswerOnlyValidClients" : single_bool,
	"netbootAnswerRequests" : single_bool,
	"netbootCurrentClientCount" : single_int,
	"netbootDUID" : single_bytes,
	"netbootGUID" : single_bytes,
	"netbootInitialization" : single_str,
	"netbootIntelliMirrorOSes" : multi_str,
	"netbootLimitClients" : single_bool,
	"netbootLocallyInstalledOSes" : multi_str,
	"netbootMachineFilePath" : single_str,
	"netbootMaxClients" : single_int,
	"netbootMirrorDataFile" : multi_str,
	"netbootNewMachineNamingPolicy" : multi_str,
	"netbootNewMachineOU" : single_str,
	"netbootSCPBL" : multi_str,
	"netbootServer" : single_str,
	"netbootSIFFile" : multi_str,
	"netbootTools" : multi_str,
	"networkAddress" : multi_str,
	"nextLevelStore" : single_str,
	"nextRid" : single_int,
	"nisMapEntry" : single_str,
	"nisMapName" : single_str,
	"nisNetgroupTriple" : multi_str,
	"nonSecurityMember" : multi_str,
	"nonSecurityMemberBL" : multi_str,
	"notificationList" : single_str,
	"nTGroupMembers" : multi_bytes,
	"nTMixedDomain" : single_int,
	"ntPwdHistory" : multi_bytes,
	"nTSecurityDescriptor" : single_sd,
	"distinguishedName" : single_str,
	"objectCategory" : single_str,
	"objectClass" : multi_str,
	"objectClassCategory" : single_int,
	"objectClasses" : multi_str,
	"objectCount" : single_int,
	"objectGUID" : single_bytes,
	"objectSid" : single_sid,
	"objectVersion" : single_int,
	"oEMInformation" : single_str,
	"oMObjectClass" : single_bytes,
	"oMSyntax" : single_int,
	"oMTGuid" : single_bytes,
	"oMTIndxGuid" : single_bytes,
	"oncRpcNumber" : single_int,
	"operatingSystem" : single_str,
	"operatingSystemHotfix" : single_str,
	"operatingSystemServicePack" : single_str,
	"operatingSystemVersion" : single_str,
	"operatorCount" : single_int,
	"optionDescription" : multi_str,
	"options" : single_int,
	"optionsLocation" : multi_str,
	"o" : multi_str,
	"ou" : multi_str,
	"organizationalStatus" : multi_str,
	"originalDisplayTable" : single_bytes,
	"originalDisplayTableMSDOS" : single_bytes,
	"otherLoginWorkstations" : multi_str,
	"otherMailbox" : multi_str,
	"middleName" : single_str,
	"otherWellKnownObjects" : multi_str,
	"owner" : single_str,
	"packageFlags" : single_int,
	"packageName" : single_str,
	"packageType" : single_int,
	"parentCA" : single_str,
	"parentCACertificateChain" : single_bytes,
	"parentGUID" : single_bytes,
	"partialAttributeDeletionList" : single_bytes,
	"partialAttributeSet" : single_bytes,
	"pekKeyChangeInterval" : single_int,
	"pekList" : single_bytes,
	"pendingCACertificates" : single_bytes,
	"pendingParentCA" : multi_str,
	"perMsgDialogDisplayTable" : single_bytes,
	"perRecipDialogDisplayTable" : single_bytes,
	"personalTitle" : single_str,
	"otherFacsimileTelephoneNumber" : multi_str,
	"otherHomePhone" : multi_str,
	"homePhone" : single_str,
	"otherIpPhone" : multi_str,
	"ipPhone" : single_str,
	"primaryInternationalISDNNumber" : single_str,
	"otherMobile" : multi_str,
	"mobile" : single_str,
	"otherTelephone" : multi_str,
	"otherPager" : multi_str,
	"pager" : single_str,
	"photo" : multi_bytes,
	"physicalDeliveryOfficeName" : single_str,
	"physicalLocationObject" : single_str,
	"thumbnailPhoto" : single_bytes,
	"pKICriticalExtensions" : multi_str,
	"pKIDefaultCSPs" : multi_str,
	"pKIDefaultKeySpec" : single_int,
	"pKIEnrollmentAccess" : multi_sd,
	"pKIExpirationPeriod" : single_bytes,
	"pKIExtendedKeyUsage" : multi_str,
	"pKIKeyUsage" : single_bytes,
	"pKIMaxIssuingDepth" : single_int,
	"pKIOverlapPeriod" : single_bytes,
	"pKT" : single_bytes,
	"pKTGuid" : single_bytes,
	"policyReplicationFlags" : single_int,
	"portName" : multi_str,
	"possSuperiors" : multi_str,
	"possibleInferiors" : multi_str,
	"postOfficeBox" : multi_str,
	"postalAddress" : multi_str,
	"postalCode" : single_str,
	"preferredDeliveryMethod" : multi_int,
	"preferredOU" : single_str,
	"preferredLanguage" : single_str,
	"prefixMap" : single_bytes,
	#"presentationAddress" : single_bstr,
	"previousCACertificates" : single_bytes,
	"previousParentCA" : multi_str,
	"primaryGroupID" : single_int,
	"primaryGroupToken" : single_int,
	"printAttributes" : single_int,
	"printBinNames" : multi_str,
	"printCollate" : single_bool,
	"printColor" : single_bool,
	"printDuplexSupported" : single_bool,
	"printEndTime" : single_int,
	"printFormName" : single_str,
	"printKeepPrintedJobs" : single_bool,
	"printLanguage" : multi_str,
	"printMACAddress" : single_str,
	"printMaxCopies" : single_int,
	"printMaxResolutionSupported" : single_int,
	"printMaxXExtent" : single_int,
	"printMaxYExtent" : single_int,
	"printMediaReady" : multi_str,
	"printMediaSupported" : multi_str,
	"printMemory" : single_int,
	"printMinXExtent" : single_int,
	"printMinYExtent" : single_int,
	"printNetworkAddress" : single_str,
	"printNotify" : single_str,
	"printNumberUp" : single_int,
	"printOrientationsSupported" : multi_str,
	"printOwner" : single_str,
	"printPagesPerMinute" : single_int,
	"printRate" : single_int,
	"printRateUnit" : single_str,
	"printSeparatorFile" : single_str,
	"printShareName" : multi_str,
	"printSpooling" : single_str,
	"printStaplingSupported" : single_bool,
	"printStartTime" : single_int,
	"printStatus" : single_str,
	"printerName" : single_str,
	"priorSetTime" : single_int,
	"priorValue" : single_bytes,
	"priority" : single_int,
	"privateKey" : single_bytes,
	"privilegeAttributes" : single_int,
	"privilegeDisplayName" : single_str,
	"privilegeHolder" : multi_str,
	"privilegeValue" : single_int,
	"productCode" : single_bytes,
	"profilePath" : single_str,
	"proxiedObjectName" : single_str,
	"proxyAddresses" : multi_str,
	"proxyGenerationEnabled" : single_bool,
	"proxyLifetime" : single_int,
	"publicKeyPolicy" : single_bytes,
	"purportedSearch" : single_str,
	"pwdHistoryLength" : single_int,
	"pwdLastSet" : single_int,
	"pwdProperties" : single_int,
	"qualityOfService" : single_int,
	"queryFilter" : single_str,
	"queryPolicyBL" : multi_str,
	"queryPolicyObject" : single_str,
	"queryPoint" : single_str,
	"rangeLower" : single_int,
	"rangeUpper" : single_int,
	"name" : single_str,
	"rDNAttID" : single_str,
	"registeredAddress" : multi_bytes,
	"remoteServerName" : multi_str,
	"remoteSource" : single_str,
	"remoteSourceType" : single_int,
	"remoteStorageGUID" : single_str,
	"replInterval" : single_int,
	"replPropertyMetaData" : single_bytes,
	"replTopologyStayOfExecution" : single_int,
	"replUpToDateVector" : single_bytes,
	"replicaSource" : single_str,
	"directReports" : multi_str,
	"repsFrom" : multi_bytes,
	"repsTo" : multi_bytes,
	"requiredCategories" : multi_bytes,
	"retiredReplDSASignatures" : single_bytes,
	"revision" : single_int,
	"rid" : single_int,
	"rIDAllocationPool" : single_int,
	"rIDAvailablePool" : single_int,
	"rIDManagerReference" : single_str,
	"rIDNextRID" : single_int,
	"rIDPreviousAllocationPool" : single_int,
	"rIDSetReferences" : multi_str,
	"rIDUsedPool" : single_int,
	"rightsGuid" : single_str,
	"roleOccupant" : multi_str,
	"roomNumber" : multi_str,
	"rootTrust" : multi_str,
	"rpcNsAnnotation" : single_str,
	"rpcNsBindings" : multi_str,
	"rpcNsCodeset" : multi_str,
	"rpcNsEntryFlags" : single_int,
	"rpcNsGroup" : multi_str,
	"rpcNsInterfaceID" : single_str,
	"rpcNsObjectID" : multi_str,
	"rpcNsPriority" : multi_int,
	"rpcNsProfileEntry" : single_str,
	"rpcNsTransferSyntax" : single_str,
	"sAMAccountName" : single_str,
	"sAMAccountType" : single_int,
	"samDomainUpdates" : single_bytes,
	"schedule" : single_bytes,
	"schemaFlagsEx" : single_int,
	"schemaIDGUID" : single_bytes,
	"schemaInfo" : multi_bytes,
	"schemaUpdate" : single_date,
	"schemaVersion" : multi_int,
	"scopeFlags" : single_int,
	"scriptPath" : single_str,
	"sDRightsEffective" : single_int,
	"searchFlags" : single_int,
	"searchGuide" : multi_bytes,
	"secretary" : multi_str,
	"securityIdentifier" : single_sid,
	"seeAlso" : multi_str,
	"seqNotification" : single_int,
	"serialNumber" : multi_str,
	"serverName" : single_str,
	"serverReference" : single_str,
	"serverReferenceBL" : multi_str,
	"serverRole" : single_int,
	"serverState" : single_int,
	"serviceBindingInformation" : multi_str,
	"serviceClassID" : single_bytes,
	"serviceClassInfo" : multi_bytes,
	"serviceClassName" : single_str,
	"serviceDNSName" : single_str,
	"serviceDNSNameType" : single_str,
	"serviceInstanceVersion" : single_bytes,
	"servicePrincipalName" : multi_str,
	"setupCommand" : single_str,
	"shadowExpire" : single_int,
	"shadowFlag" : single_int,
	"shadowInactive" : single_int,
	"shadowLastChange" : single_int,
	"shadowMax" : single_int,
	"shadowMin" : single_int,
	"shadowWarning" : single_int,
	"shellContextMenu" : multi_str,
	"shellPropertyPages" : multi_str,
	"shortServerName" : single_str,
	"showInAddressBook" : multi_str,
	"showInAdvancedViewOnly" : single_bool,
	"sIDHistory" : multi_sid,
	"signatureAlgorithms" : single_str,
	"siteGUID" : single_bytes,
	"siteLinkList" : multi_str,
	"siteList" : multi_str,
	"siteObject" : single_str,
	"siteObjectBL" : multi_str,
	"siteServer" : multi_str,
	"mailAddress" : single_str,
	"sPNMappings" : multi_str,
	"st" : single_str,
	"street" : single_str,
	"structuralObjectClass" : multi_str,
	"subClassOf" : single_str,
	"subRefs" : multi_str,
	"subSchemaSubEntry" : multi_str,
	"superScopeDescription" : multi_str,
	"superScopes" : multi_str,
	"superiorDNSRoot" : single_str,
	"supplementalCredentials" : multi_bytes,
	"supportedApplicationContext" : multi_bytes,
	"sn" : single_str,
	"syncAttributes" : single_int,
	"syncMembership" : multi_str,
	"syncWithObject" : single_str,
	"syncWithSID" : single_sid,
	"systemAuxiliaryClass" : multi_str,
	"systemFlags" : single_int,
	"systemMayContain" : multi_str,
	"systemMustContain" : multi_str,
	"systemOnly" : single_bool,
	"systemPossSuperiors" : multi_str,
	"telephoneNumber" : single_str,
	"teletexTerminalIdentifier" : multi_bytes,
	"telexNumber" : multi_bytes,
	"primaryTelexNumber" : single_str,
	"templateRoots" : multi_str,
	"templateRoots2" : multi_str,
	"terminalServer" : single_bytes,
	"co" : single_str,
	"textEncodedORAddress" : single_str,
	"timeRefresh" : single_int,
	"timeVolChange" : single_int,
	"title" : single_str,
	"tokenGroups" : multi_sid,
	"tokenGroupsGlobalAndUniversal" : multi_sid,
	"tokenGroupsNoGCAcceptable" : multi_sid,
	"tombstoneLifetime" : single_int,
	"transportAddressAttribute" : single_str,
	"transportDLLName" : single_str,
	"transportType" : single_str,
	"treatAsLeaf" : single_bool,
	"treeName" : single_str,
	"trustAttributes" : single_int,
	"trustAuthIncoming" : single_bytes,
	"trustAuthOutgoing" : single_bytes,
	"trustDirection" : single_int,
	"trustParent" : single_str,
	"trustPartner" : single_str,
	"trustPosixOffset" : single_int,
	"trustType" : single_int,
	"uASCompat" : single_int,
	"uid" : multi_str,
	"uidNumber" : single_int,
	"uNCName" : single_str,
	"unicodePwd" : single_bytes,
	"userPassword": single_str,
	"uniqueIdentifier" : multi_str,
	"uniqueMember" : multi_str,
	"unixHomeDirectory" : single_str,
	"unixUserPassword" : multi_bytes,
	"unstructuredAddress" : multi_str,
	"unstructuredName" : multi_str,
	"upgradeProductCode" : multi_bytes,
	"uPNSuffixes" : multi_str,
	"userAccountControl" : single_int,
	"userCert" : single_bytes,
	"comment" : single_str,
	"userParameters" : single_str,
	"userPassword" : multi_bytes,
	"userPrincipalName" : single_str,
	"userSharedFolder" : single_str,
	"userSharedFolderOther" : multi_str,
	"userSMIMECertificate" : multi_bytes,
	"userWorkstations" : single_str,
	"userClass" : multi_str,
	"userPKCS12" : multi_bytes,
	"uSNChanged" : single_int,
	"uSNCreated" : single_int,
	"uSNDSALastObjRemoved" : single_int,
	"USNIntersite" : single_int,
	"uSNLastObjRem" : single_int,
	"uSNSource" : single_int,
	"validAccesses" : single_int,
	"vendor" : single_str,
	"versionNumber" : single_int,
	"versionNumberHi" : single_int,
	"versionNumberLo" : single_int,
	"volTableGUID" : single_bytes,
	"volTableIdxGUID" : single_bytes,
	"volumeCount" : single_int,
	"wbemPath" : multi_str,
	"wellKnownObjects" : multi_str,
	"whenChanged" : single_date,
	"whenCreated" : single_date,
	"winsockAddresses" : multi_bytes,
	"wWWHomePage" : single_str,
	"url" : multi_str,
	"x121Address" : multi_str,
	"x500uniqueIdentifier" : multi_bytes,
	"userCertificate" : multi_bytes,
}